<?php
/**
 * WP tags and utils
 *
 * @package WordPress
 * @subpackage WINDSOR
 * @since WINDSOR 1.0
 */

// Theme init
if (!function_exists('windsor_wp_theme_setup')) {
	add_action( 'after_setup_theme', 'windsor_wp_theme_setup' );
	function windsor_wp_theme_setup() {

		// AJAX: Incremental search
		add_action('wp_ajax_ajax_search',			'windsor_callback_ajax_search');
		add_action('wp_ajax_nopriv_ajax_search',	'windsor_callback_ajax_search');

		// Filter wp_title to print a neat <title> tag based on what is being viewed
		add_filter('wp_title',						'windsor_wp_title', 10, 2);

		// Clear menu cache
		add_action('wp_update_nav_menu',			'windsor_clear_menu_cache', 10, 2);

		add_filter('comment_form_fields',			'windsor_comment_form_fields');
	}
}


/* Blog utilities
-------------------------------------------------------------------------------- */

// Detect current blog mode to get correspond options (post | page | search | blog | home)
if (!function_exists('windsor_detect_blog_mode')) {
	function windsor_detect_blog_mode() {
		if (is_front_page())
			$mode = 'home';
		else if (is_single())
			$mode = 'post';
		else if (is_page() && !windsor_storage_isset('blog_archive'))
			$mode = 'page';
		else
			$mode = 'blog';
		return apply_filters('windsor_filter_detect_blog_mode', $mode);
	}
}


// Return current site protocol
if (!function_exists('windsor_get_protocol')) {
	function windsor_get_protocol() {
		return is_ssl() ? 'https' : 'http';
	}
}

// Return internal page link - if is customize mode - full url else only hash part
if (!function_exists('windsor_get_hash_link')) {
	function windsor_get_hash_link($hash) {
		if (strpos($hash, 'http')!==0) {
			if ($hash[0]!='#') $hash = '#'.$hash;
			if (is_customize_preview()) {
				$url = windsor_get_current_url();
				if (($pos=strpos($url, '#'))!==false) $url = substr($url, 0, $pos);
				$hash = $url . $hash;
			}
		}
		return $hash;
	}
}

// Return URL to the current page
if (!function_exists('windsor_get_current_url')) {
	function windsor_get_current_url() {
		global $wp;
		// Attention! We don't need to process it with esc_url()
		// since this url is being processed with esc_url() where it's used.
		return home_url(add_query_arg(array(), $wp->request));
	}
}

// Filters wp_title to print a neat <title> tag based on what is being viewed.
if ( !function_exists( 'windsor_wp_title' ) ) {
	function windsor_wp_title( $title, $sep ) {
		if ( is_feed() ) return $title;
		if (floatval(get_bloginfo('version')) < "4.1") {
			global $page, $paged;
			// Add the blog name
			$title .= get_bloginfo( 'name' );
			// Add the blog description for the home/front page.
			if ( is_home() || is_front_page() ) {
				if ( ($site_description = get_bloginfo( 'description', 'display' )) != '' )
					$title .= " $sep $site_description";
			}
			// Add a page number if necessary:
			if ( $paged >= 2 || $page >= 2 )
				$title .= " $sep " . sprintf( esc_html__( 'Page %s', 'windsor' ), max( $paged, $page ) );
		}
		return windsor_remove_macros($title);
	}
}

// Return blog title
if (!function_exists('windsor_get_blog_title')) {
	function windsor_get_blog_title() {

		if (is_front_page())
			$title = esc_html__( 'Home', 'windsor' );
		else if ( is_home() )
			$title = esc_html__( 'All Posts', 'windsor' );
		else if ( is_author() ) {
			$curauth = (get_query_var('author_name')) ? get_user_by('slug', get_query_var('author_name')) : get_userdata(get_query_var('author'));
			$title = sprintf(esc_html__('Author page: %s', 'windsor'), $curauth->display_name);
		} else if ( is_404() )
			$title = esc_html__('URL not found', 'windsor');
		else if ( is_search() )
			$title = sprintf( esc_html__( 'Search: %s', 'windsor' ), get_search_query() );
		else if ( is_day() )
			$title = sprintf( esc_html__( 'Daily Archives: %s', 'windsor' ), get_the_date() );
		else if ( is_month() )
			$title = sprintf( esc_html__( 'Monthly Archives: %s', 'windsor' ), get_the_date( 'F Y' ) );
		else if ( is_year() )
			$title = sprintf( esc_html__( 'Yearly Archives: %s', 'windsor' ), get_the_date( 'Y' ) );
		 else if ( is_category() )
			$title = sprintf( '%s' , single_cat_title( '', false ) );
		else if ( is_tag() )
			$title = sprintf( esc_html__( 'Tag: %s', 'windsor' ), single_tag_title( '', false ) );
		else if ( is_tax() )
			$title = sprintf( '%s', single_term_title( '', false ) );
		else if ( is_attachment() )
			$title = sprintf( esc_html__( 'Attachment: %s', 'windsor' ), get_the_title());
		else if ( is_single() || is_page() )
			$title = get_the_title();
		else
			$title = get_the_title();
		return apply_filters('windsor_filter_get_blog_title', $title);
	}
}

// Show breadcrumbs path
if (!function_exists('windsor_get_breadcrumbs')) {
	function windsor_get_breadcrumbs($args=array()) {
		global $wp_query, $post;

		$args = array_merge( array(
			'home' => esc_html__('Home', 'windsor'),		// Home page title (if empty - not showed)
			'home_link' => '',								// Home page link
			'truncate_title' => 50,					// Truncate all titles to this length (if 0 - no truncate)
			'truncate_add' => '...',				// Append truncated title with this string
			'delimiter' => '<span class="breadcrumbs_delimiter"></span>',			// Delimiter between breadcrumbs items
			'max_levels' => windsor_get_theme_setting('breadcrumbs_max_level'),		// Max categories in the path (0 - unlimited)
			), is_array($args) ? $args : array( 'home' => $args )
		);

		if ( is_front_page() ) return '';

		if ( $args['max_levels']<=0 ) $args['max_levels'] = 999;

		$need_reset = true;
		$rez = $rez_parent = $rez_level = '';
		$cat = $parent_tax = '';
		$level = $parent = $post_id = 0;

		// Get current post ID and path to current post/page/attachment ( if it have parent posts/pages )
		if (is_page() || is_attachment() || is_single()) {
			$page_parent_id = apply_filters('windsor_filter_get_parent_id', isset($wp_query->post->post_parent) ? $wp_query->post->post_parent : 0, isset($wp_query->post->ID) ? $wp_query->post->ID : 0);
			$post_id = (is_attachment() ? $page_parent_id : (isset($wp_query->post->ID) ? $wp_query->post->ID : 0));
			while ($page_parent_id > 0) {
				$page_parent = get_post($page_parent_id);
				$level++;
				if ($level > $args['max_levels'])
					$rez_level = '...';
				else
					$rez_parent = '<a class="breadcrumbs_item cat_post" href="' . esc_url(get_permalink($page_parent->ID)) . '">'
									. esc_html(windsor_strshort($page_parent->post_title, $args['truncate_title'], $args['truncate_add']))
									. '</a>'
									. (!empty($rez_parent) ? $args['delimiter'] : '')
									. ($rez_parent);
				if (($page_parent_id = apply_filters('windsor_filter_get_parent_id', $page_parent->post_parent, $page_parent_id)) > 0) $post_id = $page_parent_id;
			}
		}

		// Show parents
		$step = 0;
		do {
			if ($step++ == 0) {
				if (is_single() || is_attachment()) {
					$post_type = get_post_type();
					if ($post_type == 'post') {
						$cats = get_the_category();
						$cat = !empty($cats[0]) ? $cats[0] : false;
					} else {
						$tax = apply_filters('windsor_filter_post_type_taxonomy', '', $post_type);
						if (!empty($tax)) {
							$cats = get_the_terms(get_the_ID(), $tax);
							$cat = !empty($cats[0]) ? $cats[0] : false;
						}
					}
					if ($cat) {
						$level++;
						if ($level > $args['max_levels'])
							$rez_level = '...';
						else
							$rez_parent = '<a class="breadcrumbs_item cat_post" href="'.esc_url(get_category_link($cat->term_id)).'">'
											. esc_html(windsor_strshort($cat->name, $args['truncate_title'], $args['truncate_add']))
											. '</a>'
											. (!empty($rez_parent) ? $args['delimiter'] : '')
											. ($rez_parent);
					}
				} else if ( is_category() ) {
					$cat_id = (int) get_query_var( 'cat' );
					if (empty($cat_id)) $cat_id = get_query_var( 'category_name' );
					$cat = get_term_by( (int) $cat_id > 0 ? 'id' : 'slug', $cat_id, 'category', OBJECT);
				} else if ( is_tag() ) {
					$cat = get_term_by( 'slug', get_query_var( 'post_tag' ), 'post_tag', OBJECT);
				} else if ( is_tax() ) {
					$tax = get_query_var('taxonomy');
					$cat = get_term_by( 'slug', get_query_var( $tax ), $tax, OBJECT);
				}
				if ($cat) {
					$parent = $cat->parent;
					$parent_tax = $cat->taxonomy;
				}
			}
			if ($parent) {
				$cat = get_term_by( 'id', $parent, $parent_tax, OBJECT);
				if ($cat) {
					$cat_link = get_term_link($cat->slug, $cat->taxonomy);
					$level++;
					if ($level > $args['max_levels'])
						$rez_level = '...';
					else
						$rez_parent = '<a class="breadcrumbs_item cat_parent" href="'.esc_url($cat_link).'">'
										. esc_html(windsor_strshort($cat->name, $args['truncate_title'], $args['truncate_add']))
										. '</a>'
										. (!empty($rez_parent) ? $args['delimiter'] : '')
										. ($rez_parent);
					$parent = $cat->parent;
				}
			}
		} while ($parent);

		$rez_period = '';
		if ((is_day() || is_month()) && is_object($post)) {
			$year  = get_the_time('Y');
			$month = get_the_time('m');
			$rez_period = '<a class="breadcrumbs_item cat_parent" href="' . esc_url(get_year_link( $year )) . '">' . ($year) . '</a>';
			if (is_day())
				$rez_period .= (!empty($rez_period) ? $args['delimiter'] : '') . '<a class="breadcrumbs_item cat_parent" href="' . esc_url(get_month_link( $year, $month )) . '">' . esc_html(get_the_date('F')) . '</a>';
		}

		// Get link to the 'All posts (products, events, etc.)' page
		$rez_all = apply_filters('windsor_filter_get_blog_all_posts_link', '');

		if (!is_front_page()) {

			$title = windsor_get_blog_title();
			if (is_array($title)) $title = $title['text'];
			$title = windsor_strshort($title, $args['truncate_title'], $args['truncate_add']);

			$rez .= (isset($args['home']) && $args['home']!=''
					? '<a class="breadcrumbs_item home" href="' . esc_url($args['home_link'] ? $args['home_link'] : home_url('/')) . '">' . ($args['home']) . '</a>' . ($args['delimiter'])
					: '')
				. (!empty($rez_all)    ? ($rez_all)    . ($args['delimiter']) : '')
				. (!empty($rez_level)  ? ($rez_level)  . ($args['delimiter']) : '')
				. (!empty($rez_parent) ? ($rez_parent) . ($args['delimiter']) : '')
				. (!empty($rez_period) ? ($rez_period) . ($args['delimiter']) : '')
				. ($title ? '<span class="breadcrumbs_item current">' . ($title) . '</span>' : '');
		}

		return apply_filters('windsor_filter_get_breadcrumbs', $rez);
	}
}

// Return nav menu html
if ( !function_exists( 'windsor_get_nav_menu' ) ) {
	function windsor_get_nav_menu($slug='', $depth=11, $custom_walker=false) {
		$menu = '';
		$menu_cache = windsor_is_on(windsor_get_theme_option('menu_cache')) ? 'windsor_menu_'.get_option('stylesheet') : '';
		$list = $menu_cache ? get_transient($menu_cache) : array();
		if (!is_array($list)) $list = array();
		$html = '';
		if (!empty($slug) && !empty($list[$slug])) {
			$html = $list[$slug];
		}
		if (empty($html)) {
			$args = array(
				'menu'				=> empty($menu) || $menu=='default' || windsor_is_inherit($menu) ? '' : $menu,
				'container'			=> '',
				'container_class'	=> '',
				'container_id'		=> '',
				'items_wrap'		=> '<ul id="%1$s" class="%2$s">%3$s</ul>',
				'menu_class'		=> (!empty($slug) ? $slug : 'menu_main') . '_nav',
				'menu_id'			=> (!empty($slug) ? $slug : 'menu_main'),
				'echo'				=> false,
				'fallback_cb'		=> '',
				'before'			=> '',
				'after'				=> '',
				'link_before'       => '<span>',
				'link_after'        => '</span>',
				'depth'             => $depth
			);
			if (!empty($slug))
				$args['theme_location'] = $slug;
			if ($custom_walker && class_exists('windsor_custom_menu_walker'))
				$args['walker'] = new windsor_custom_menu_walker;
			$html = wp_nav_menu(apply_filters('windsor_filter_get_nav_menu_args', $args));
			if (!empty($slug) && $menu_cache) {
				$list[$slug] = $html;
				set_transient($menu_cache, $list, 24*60*60);
			}
		}
		return apply_filters('windsor_filter_get_nav_menu', $html);
	}
}

// Clear cache with saved menu
if ( !function_exists( 'windsor_clear_menu_cache' ) ) {
	function windsor_clear_menu_cache($menu_id=0, $menu_data=array()) {
		delete_transient('windsor_menu_'.get_option('stylesheet'));
	}
}

// AJAX incremental search
if ( !function_exists( 'windsor_callback_ajax_search' ) ) {
	function windsor_callback_ajax_search() {
		if ( !wp_verify_nonce( windsor_get_value_gp('nonce'), admin_url('admin-ajax.php') ) )
            wp_die();

		$response = array('error'=>'', 'data' => '');

		$s = sanitize_text_field($_REQUEST['text']);

		if (!empty($s)) {
			$args = apply_filters( 'windsor_ajax_search_query', array(
				'post_status' => 'publish',
				'orderby' => 'date',
				'order' => 'desc',
				'posts_per_page' => 4,
				's' => esc_html($s),
				)
			);

			$post_number = 0;
			windsor_storage_set('windsor_output_widgets_posts', '');
			$query = new WP_Query( $args );
			set_query_var('windsor_args_widgets_posts', array(
				'show_image' => 1,
				'show_date' => 1,
				'show_author' => 1,
				'show_counters' => 1,
                'show_categories' => 0
   	            )
       	    );
			while ( $query->have_posts() ) { $query->the_post();
				$post_number++;
				get_template_part('templates/widgets-posts');
			}
			$response['data'] = windsor_storage_get('windsor_output_widgets_posts');
			if (empty($response['data'])) {
				$response['data'] .= '<article class="post_item">' . esc_html__('Sorry, but nothing matched your search criteria. Please try again with some different keywords.', 'windsor') . '</article>';
			} else {
				$response['data'] .= '<article class="post_item"><a href="#" class="post_more search_more">' . esc_html__('More results ...', 'windsor') . '</a></article>';
			}
			windsor_storage_set('output', '');
		} else {
			$response['error'] = '<article class="post_item">' . esc_html__('The query string is empty!', 'windsor') . '</article>';
		}

		echo json_encode($response);
        wp_die();
	}
}

// Return string with categories links
if (!function_exists('windsor_get_post_categories')) {
	function windsor_get_post_categories($delimiter=', ', $id=false) {
		$output = '';
		$categories = get_the_category($id);
		if ( !empty( $categories ) ) {
			foreach( $categories as $category )
				$output .= ($output ? $delimiter : '') . '<a href="' . esc_url( get_category_link( $category->term_id ) ) . '" title="' . sprintf( esc_attr__( 'View all posts in %s', 'windsor' ), $category->name ) . '">' . esc_html( $category->name ) . '</a>';
		}
		return $output;
	}
}

// Return string with terms links
if (!function_exists('windsor_get_post_terms')) {
	function windsor_get_post_terms($delimiter=', ', $id=false, $taxonomy='category') {
		$output = '';
		$terms = get_the_terms($id, $taxonomy);
		if ( !empty( $terms ) ) {
			foreach( $terms as $term )
				$output .= ($output ? $delimiter : '') . '<a href="' . esc_url( get_term_link( $term->term_id, $taxonomy ) ) . '" title="' . sprintf( esc_attr__( 'View all posts in %s', 'windsor' ), esc_attr($term->name) ) . '">' . esc_html( $term->name ) . '</a>';
		}
		return $output;
	}
}

// Comment form fields order
if ( !function_exists( 'windsor_comment_form_fields' ) ) {
	
	function windsor_comment_form_fields($comment_fields) {
		if (windsor_get_theme_setting('comment_after_name')) {
			$keys = array_keys($comment_fields);
			if ($keys[0]=='comment') $comment_fields['comment'] = array_shift($comment_fields);
		}
		return $comment_fields;
	}
}


/* Query manipulations
-------------------------------------------------------------------------------- */

// Add sorting parameter in query arguments
if (!function_exists('windsor_query_add_sort_order')) {
	function windsor_query_add_sort_order($args, $orderby='date', $order='desc') {
		$q = apply_filters('windsor_filter_query_sort_order', array(), $orderby, $order);
		$q['order'] = $order=='asc' ? 'asc' : 'desc';
		if (empty($q['orderby'])) {
			if ($orderby == 'comments') {
				$q['orderby'] = 'comment_count';
			} else if ($orderby == 'title' || $orderby == 'alpha') {
				$q['orderby'] = 'title';
			} else if ($orderby == 'rand' || $orderby == 'random')  {
				$q['orderby'] = 'rand';
			} else {
				$q['orderby'] = 'post_date';
			}
		}
		foreach ($q as $mk=>$mv) {
			if (is_array($args))
				$args[$mk] = $mv;
			else
				$args->set($mk, $mv);
		}
		return $args;
	}
}

// Add post type and posts list or categories list in query arguments
if (!function_exists('windsor_query_add_posts_and_cats')) {
	function windsor_query_add_posts_and_cats($args, $ids='', $post_type='', $cat='', $taxonomy='category') {
		if (!empty($ids)) {
			$args['post_type'] = empty($args['post_type'])
									? (empty($post_type) ? array('post', 'page') : $post_type)
									: $args['post_type'];
			$args['post__in'] = explode(',', str_replace(' ', '', $ids));
		} else {
			$args['post_type'] = empty($args['post_type'])
									? (empty($post_type) ? 'post' : $post_type)
									: $args['post_type'];
			$post_type = is_array($args['post_type']) ? $args['post_type'][0] : $args['post_type'];
			if (!empty($cat)) {
				$cats = !is_array($cat) ? explode(',', $cat) : $cat;
				if ($taxonomy == 'category') {				// Add standard categories
					if (is_array($cats) && count($cats) > 1) {
						$cats_ids = array();
						foreach($cats as $c) {
							$c = trim($c);
							if (empty($c)) continue;
							if ((int) $c == 0) {
								$cat_term = get_term_by( 'slug', $c, $taxonomy, OBJECT);
								if ($cat_term) $c = $cat_term->term_id;
							}
							if ($c==0) continue;
							$cats_ids[] = (int) $c;
							$children = get_categories( array(
								'type'                     => $post_type,
								'child_of'                 => $c,
								'hide_empty'               => 0,
								'hierarchical'             => 0,
								'taxonomy'                 => $taxonomy,
								'pad_counts'               => false
							));
							if (is_array($children) && count($children) > 0) {
								foreach($children as $c) {
									if (!in_array((int) $c->term_id, $cats_ids)) $cats_ids[] = (int) $c->term_id;
								}
							}
						}
						if (count($cats_ids) > 0) {
							$args['category__in'] = $cats_ids;
						}
					} else {
						if ((int) $cat > 0)
							$args['cat'] = (int) $cat;
						else
							$args['category_name'] = $cat;
					}
				} else {									// Add custom taxonomies
					if (!isset($args['tax_query']))
						$args['tax_query'] = array();
					$args['tax_query']['relation'] = 'AND';
					$args['tax_query'][] = array(
						'taxonomy' => $taxonomy,
						'include_children' => true,
						'field'    => (int) $cats[0] > 0 ? 'id' : 'slug',
						'terms'    => $cats
					);
				}
			}
		}
		return $args;
	}
}

// Add filters (meta parameters) in query arguments
if (!function_exists('windsor_query_add_filters')) {
	function windsor_query_add_filters($args, $filters=false) {
		if (!empty($filters)) {
			if (!is_array($filters)) $filters = array($filters);
			foreach ($filters as $v) {
				$found = false;
				if ($v=='thumbs') {							// Filter with meta_query
					if (!isset($args['meta_query']))
						$args['meta_query'] = array();
					else {
						for ($i=0; $i<count($args['meta_query']); $i++) {
							if ($args['meta_query'][$i]['meta_filter'] == $v) {
								$found = true;
								break;
							}
						}
					}
					if (!$found) {
						$args['meta_query']['relation'] = 'AND';
						if ($v == 'thumbs') {
							$args['meta_query'][] = array(
								'meta_filter' => $v,
								'key' => '_thumbnail_id',
								'value' => false,
								'compare' => '!='
							);
						}
					}
				} else if (in_array($v, array('video', 'audio', 'gallery'))) {			// Filter with tax_query
					if (!isset($args['tax_query']))
						$args['tax_query'] = array();
					else {
						for ($i=0; $i<count($args['tax_query']); $i++) {
							if ($args['tax_query'][$i]['tax_filter'] == $v) {
								$found = true;
								break;
							}
						}
					}
					if (!$found) {
						$args['tax_query']['relation'] = 'AND';
						if ($v == 'video') {
							$args['tax_query'][] = array(
								'tax_filter' => $v,
								'taxonomy' => 'post_format',
								'field' => 'slug',
								'terms' => array( 'post-format-video' )
							);
						} else if ($v == 'audio') {
							$args['tax_query'] = array(
								'tax_filter' => $v,
								'taxonomy' => 'post_format',
								'field' => 'slug',
								'terms' => array( 'post-format-audio' )
							);
						} else if ($v == 'gallery') {
							$args['tax_query'] = array(
								'tax_filter' => $v,
								'taxonomy' => 'post_format',
								'field' => 'slug',
								'terms' => array( 'post-format-gallery' )
							);
						}
					}
				}
			}
		}
		return $args;
	}
}




/* Widgets utils
------------------------------------------------------------------------------------- */

// Create widgets area
if (!function_exists('windsor_create_widgets_area')) {
	function windsor_create_widgets_area($name, $add_classes='') {
		$widgets_name = windsor_get_theme_option($name);
		if (!windsor_is_off($widgets_name) && is_active_sidebar($widgets_name)) {
			set_query_var('current_sidebar', $name);
			ob_start();
			do_action( 'windsor_action_before_sidebar' );
            if ( is_active_sidebar( $widgets_name ) ) {
                dynamic_sidebar( $widgets_name );
            }
			do_action( 'windsor_action_after_sidebar' );
			$out = ob_get_contents();
			ob_end_clean();
			$out = preg_replace("/<\/aside>[\r\n\s]*<aside/", "</aside><aside", $out);
			$need_columns = strpos($out, 'columns_wrap')===false;
			if ($need_columns) {
				$columns = min(3, max(1, substr_count($out, '<aside ')));
				$out = preg_replace("/class=\"widget /", "class=\"column-1_".esc_attr($columns).' widget ', $out);
			}
			?>
			<div class="<?php echo esc_attr($name); ?> <?php echo esc_attr($name); ?>_wrap widget_area">
				<div class="<?php echo esc_attr($name); ?>_inner <?php echo esc_attr($name); ?>_wrap_inner widget_area_inner">
					<?php
					windsor_show_layout($out,
										true==$need_columns ? '<div class="columns_wrap">' : '',
                                        true==$need_columns ? '</div>' : ''
                                        );
					?>
				</div> <!-- /.widget_area_inner -->
			</div> <!-- /.widget_area -->
			<?php
		}
	}
}

// Check if sidebar present
if (!function_exists('windsor_sidebar_present')) {
	function windsor_sidebar_present() {
		global $wp_query;
		$sidebar_name = windsor_get_theme_option('sidebar_widgets');
		return apply_filters('windsor_filter_sidebar_present',
					!is_404()
					&& (!is_search() || $wp_query->found_posts > 0)
					&& !windsor_is_off($sidebar_name)
					&& is_active_sidebar($sidebar_name)
					);
	}
}

// Return text for the Privacy Policy checkbox
if ( ! function_exists('windsor_get_privacy_text' ) ) {
    function windsor_get_privacy_text() {
        $page = get_option( 'wp_page_for_privacy_policy' );
        $privacy_text = windsor_get_theme_option( 'privacy_text' );
        return apply_filters( 'windsor_filter_privacy_text', wp_kses_post(
                $privacy_text
                . ( ! empty( $page ) && ! empty( $privacy_text )
                    // Translators: Add url to the Privacy Policy page
                    ? ' ' . sprintf( __( 'For further details on handling user data, see our %s', 'windsor' ),
                        '<a href="' . esc_url( get_permalink( $page ) ) . '" target="_blank">'
                        . __( 'Privacy Policy', 'windsor' )
                        . '</a>' )
                    : ''
                )
            )
        );
    }
}


/* wp_kses handlers
----------------------------------------------------------------------------------------------------- */
if ( ! function_exists( 'windsor_kses_allowed_html' ) ) {
	add_filter( 'wp_kses_allowed_html', 'windsor_kses_allowed_html', 10, 2);
	function windsor_kses_allowed_html($tags, $context) {
		if ( in_array( $context, array( 'windsor_kses_content') ) ) {
			$tags = array(
				'h1'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'h2'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'h3'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'h4'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'h5'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'h6'     => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'p'      => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'span'   => array( 'id' => array(), 'class' => array(), 'title' => array() ),
				'div'    => array( 'id' => array(), 'class' => array(), 'title' => array(), 'align' => array() ),
				'a'      => array( 'id' => array(), 'class' => array(), 'title' => array(), 'href' => array(), 'target' => array() ),
				'b'      => array( 'id' => array(), 'class' => array(), 'title' => array() ),
				'i'      => array( 'id' => array(), 'class' => array(), 'title' => array() ),
				'em'     => array( 'id' => array(), 'class' => array(), 'title' => array() ),
				'strong' => array( 'id' => array(), 'class' => array(), 'title' => array() ),
				'img'    => array( 'id' => array(), 'class' => array(), 'src' => array(), 'width' => array(), 'height' => array(), 'alt' => array() ),
				'br'     => array( 'clear' => array() ),
			);
		}
		return $tags;
	}
}


/* Inline styles and scripts
------------------------------------------------------------------------------------- */

// Add inline styles and return class for it
if (!function_exists('windsor_sidebar_presen')) {
	function windsor_add_inline_style($css) {
		static $class_number = 0;
		$class_name = sprintf('windsor_inline_%d', $class_number++);
		windsor_storage_concat( 'inline_styles', sprintf('.%s{%s}', $class_name, $css) );
		return $class_name;
	}
}

/* Date & Time
----------------------------------------------------------------------------------------------------- */

// Return post date
if (!function_exists('windsor_get_date')) {
	function windsor_get_date($dt='', $format='') {
		if ($dt == '') $dt = get_the_time('U');
		if (date('U') - $dt > intval(windsor_get_theme_option('time_diff_before'))*24*3600)
			$dt = date_i18n($format=='' ? get_option('date_format') : $format, $dt);
		else
			$dt = sprintf( esc_html__('%s ago', 'windsor'), human_time_diff($dt) );
		return $dt;
	}
}



// Return full content of the post/page
if ( ! function_exists( 'windsor_get_post_content' ) ) {
	function windsor_get_post_content( $apply_filters=false ) {
		global $post;
		return $apply_filters ? apply_filters( 'the_content', $post->post_content ) : $post->post_content;
	}
}

?>